<?php
session_start();
include 'includes/db.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Fetch user details
$query = "SELECT name, email FROM users WHERE id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();

if (!$user) {
    die("User not found.");
}

// Fetch booking details
$bookings = [];
$query = "SELECT room_id, check_in, check_out, total_price, status FROM booking WHERE user_id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $bookings[] = $row;
}

$stmt->close();
$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Dashboard</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100">

    <div class="flex">
        <!-- Sidebar -->
        <div class="w-1/5 bg-gray-900 text-white min-h-screen p-5">
            <h2 class="text-2xl font-bold mb-5">User Dashboard</h2>
            <ul class="space-y-4">
                <li><a href="user_dashboard.php" class="block py-2 px-4 bg-gray-700 rounded-lg">Dashboard</a></li>
                <li><a href="accomodations.php" class="block py-2 px-4 hover:bg-gray-700 rounded-lg">Book Now</a></li>
                <li><a href="edit-profile.php" class="block py-2 px-4 hover:bg-gray-700 rounded-lg">Edit Profile</a></li>
                <li><a href="checkout.php" class="block py-2 px-4 hover:bg-gray-700 rounded-lg">Checkout</a></li>
                <li><a href="logout.php" class="block py-2 px-4 bg-red-600 rounded-lg">Logout</a></li>
            </ul>
        </div>

        <!-- Main Content -->
        <div class="w-4/5 p-8">
            <h1 class="text-3xl font-bold mb-6">Welcome, <?php echo htmlspecialchars($user['name']); ?>!</h1>

            <!-- Booking Details -->
            <div class="bg-white p-6 rounded-lg shadow-md">
                <h2 class="text-xl font-semibold mb-4">Your Bookings</h2>
                
                <?php if (!empty($bookings)) : ?>
                    <div class="overflow-x-auto">
                        <table class="min-w-full bg-gray-100 border border-gray-300 rounded-lg">
                            <thead>
                                <tr class="bg-gray-800 text-white">
                                    <th class="py-3 px-6 text-left">Room Id</th>
                                    <th class="py-3 px-6 text-left">Check-In</th>
                                    <th class="py-3 px-6 text-left">Check-Out</th>
                                    <th class="py-3 px-6 text-left">Total Price</th>
                                    <th class="py-3 px-6 text-left">Status</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-700">
                                <?php foreach ($bookings as $booking) : ?>
                                    <tr class="border-b border-gray-300">
                                        <td class="py-3 px-6"><?php echo htmlspecialchars($booking['room_id']); ?></td>
                                        <td class="py-3 px-6"><?php echo htmlspecialchars($booking['check_in']); ?></td>
                                        <td class="py-3 px-6"><?php echo htmlspecialchars($booking['check_out']); ?></td>
                                        <td class="py-3 px-6">Ksh <?php echo number_format($booking['total_price']); ?></td>
                                        <td class="py-3 px-6 font-bold <?php echo ($booking['status'] == 'Confirmed') ? 'text-green-600' : 'text-yellow-600'; ?>">
                                            <?php echo htmlspecialchars($booking['status']); ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else : ?>
                    <p class="text-gray-500 mt-4">No bookings found.</p>
                <?php endif; ?>
            </div>

            <!-- Profile Details -->
            <div class="bg-white p-6 rounded-lg shadow-md mt-6">
                <h2 class="text-xl font-semibold mb-4">Profile Settings</h2>
                <p><strong>Email:</strong> <?php echo htmlspecialchars($user['email']); ?></p>
                <a href="edit-profile.php" class="mt-4 inline-block bg-blue-600 text-white px-4 py-2 rounded-lg">Edit Profile</a>
            </div>
        </div>
    </div>

    <!-- Support Chat Button -->
    <div class="fixed bottom-5 right-5">
        <button class="bg-yellow-500 text-white px-6 py-3 rounded-full shadow-lg">Chat Support</button>
    </div>

</body>
</html>
