<?php
session_start();
require 'includes/db.php';
require 'fpdf186/fpdf.php'; // Include FPDF Library

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Fetch user details
$query = "SELECT name, email, phone FROM users WHERE id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user_result = $stmt->get_result();
$user = $user_result->fetch_assoc();
$stmt->close();

// Fetch latest booking details
$query = "SELECT b.id AS booking_id, b.room_id, b.check_in, b.check_out, b.total_price, 
                 a.name AS room_name, a.room_type 
          FROM booking b 
          JOIN accommodations a ON b.room_id = a.id 
          WHERE b.user_id = ? 
          ORDER BY b.id DESC LIMIT 1";

$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$booking_result = $stmt->get_result();
$booking = $booking_result->fetch_assoc();
$stmt->close();

if (!$booking) {
    die("No bookings found.");
}

// Check if payment already exists for this booking
$query = "SELECT id FROM payments WHERE booking_id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $booking['booking_id']);
$stmt->execute();
$stmt->store_result();

if ($stmt->num_rows > 0) {
    $stmt->close();
    echo "<script>
        alert('Payment already exists for this booking! Redirecting to dashboard...');
        setTimeout(function() { window.location.href = 'dashboard.php'; }, 3000);
    </script>";
    exit();
}
$stmt->close();

// Handle Payment Submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $payment_method = $_POST['payment_method'];
    $transaction_id = ($payment_method === 'Mpesa') ? $_POST['transaction_id'] : 'Cash Payment';
    $amount = $booking['total_price'];
    $payment_status = "Completed";

    // Insert into payments table
    $query = "INSERT INTO payments (user_id, booking_id, room_id, payment_method, amount, payment_status, transaction_id)
              VALUES (?, ?, ?, ?, ?, ?, ?)";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("iiissss", $user_id, $booking['booking_id'], $booking['room_id'], $payment_method, $amount, $payment_status, $transaction_id);
    
    if ($stmt->execute()) {
        $stmt->close();
        
        // Generate receipt PDF
        $receipt_file = generateReceipt($user, $booking, $payment_method, $transaction_id);

        echo "<script>
        alert('Payment Successful! Your receipt has been downloaded successfully.\\nLocation: receipts/');
        window.location='dashboard.php';
  </script>";
exit();


    } else {
        echo "<p class='text-red-600 text-center'>Error: " . $conn->error . "</p>";
    }
    $stmt->close();
}
$conn->close();

// Function to generate PDF receipt
function generateReceipt($user, $booking, $payment_method, $transaction_id) {
    $pdf = new FPDF();
    $pdf->AddPage();
    $pdf->SetFont('Arial', 'B', 16);
    $pdf->Cell(190, 10, 'Hotel Receipt', 0, 1, 'C');
    $pdf->Ln(5);
    
    $pdf->SetFont('Arial', '', 12);
    $pdf->Cell(100, 10, 'Customer Name: ' . htmlspecialchars($user['name']), 0, 1);
    $pdf->Cell(100, 10, 'Email: ' . htmlspecialchars($user['email']), 0, 1);
    $pdf->Cell(100, 10, 'Phone: ' . htmlspecialchars($user['phone']), 0, 1);
    $pdf->Ln(5);
    
    $pdf->Cell(100, 10, 'Room: ' . htmlspecialchars($booking['room_name']) . ' (' . htmlspecialchars($booking['room_type']) . ')', 0, 1);
    $pdf->Cell(100, 10, 'Check-in: ' . htmlspecialchars($booking['check_in']), 0, 1);
    $pdf->Cell(100, 10, 'Check-out: ' . htmlspecialchars($booking['check_out']), 0, 1);
    $pdf->Ln(5);
    
    $pdf->Cell(100, 10, 'Payment Method: ' . htmlspecialchars($payment_method), 0, 1);
    $pdf->Cell(100, 10, 'Transaction ID: ' . htmlspecialchars($transaction_id), 0, 1);
    $pdf->Cell(100, 10, 'Total Amount: Ksh ' . number_format($booking['total_price']), 0, 1);
    $pdf->Ln(10);
    
    $pdf->SetFont('Arial', 'I', 10);
    $pdf->Cell(190, 10, 'Thank you for booking with us!', 0, 1, 'C');

    $file_name = 'receipts/receipt_' . $user['id'] . '_' . time() . '.pdf';
    $pdf->Output('F', $file_name);
    return $file_name;
}
?>



<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Checkout | Alex Resort</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100">

<div class="max-w-2xl mx-auto mt-10 bg-white p-6 rounded-lg shadow-md">
    <h1 class="text-2xl font-bold mb-6 text-center">Checkout</h1>

    <div class="mb-6">
        <h2 class="text-lg font-semibold">Your Details</h2>
        <p><strong>Name:</strong> <?php echo htmlspecialchars($user['name']); ?></p>
        <p><strong>Email:</strong> <?php echo htmlspecialchars($user['email']); ?></p>
        <p><strong>Phone:</strong> <?php echo htmlspecialchars($user['phone']); ?></p>
    </div>

    <div class="mb-6">
        <h2 class="text-lg font-semibold">Booking Details</h2>
        <p><strong>Room:</strong> <?php echo htmlspecialchars($booking['room_name']); ?> (<?php echo htmlspecialchars($booking['room_type']); ?>)</p>
        <p><strong>Check-in:</strong> <?php echo htmlspecialchars($booking['check_in']); ?></p>
        <p><strong>Check-out:</strong> <?php echo htmlspecialchars($booking['check_out']); ?></p>
        <p><strong>Total Price:</strong> Ksh <?php echo number_format($booking['total_price']); ?></p>
    </div>

    <form method="POST">
        <h2 class="text-lg font-semibold mb-2">Payment Method</h2>
        <div class="mb-4">
            <input type="radio" id="mpesa" name="payment_method" value="Mpesa" required>
            <label for="mpesa" class="ml-2">Mpesa</label>
        </div>

        <div class="mb-4 hidden" id="mpesa_transaction">
            <label class="block mb-1">Mpesa Transaction ID:</label>
            <input type="text" name="transaction_id" class="w-full p-2 border rounded-lg" placeholder="Enter Transaction ID">
        </div>

        <div class="mb-4">
            <input type="radio" id="cash" name="payment_method" value="Cash" required>
            <label for="cash" class="ml-2">Cash (Pay at Hotel)</label>
        </div>

        <button type="submit" class="w-full bg-blue-600 text-white py-3 rounded-lg font-bold hover:bg-blue-700 transition">
            Confirm Payment
        </button>
    </form>
</div>

</body>
</html>
