
<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
session_start();
include 'includes/db.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$selected_room_id = isset($_GET['room_id']) ? (int) $_GET['room_id'] : null;

// Fetch user details
$user_query = $conn->prepare("SELECT name, phone, email FROM users WHERE id = ?");
$user_query->bind_param("i", $user_id);
$user_query->execute();
$user_result = $user_query->get_result();
$user = $user_result->fetch_assoc();
$user_query->close();


// Fetch room details
$rooms = [];
$selected_room = null;

$result = $conn->query("SELECT id, name, price, room_type FROM accommodations");
while ($row = $result->fetch_assoc()) {
    $rooms[] = $row;
    if ($selected_room_id && $row['id'] == $selected_room_id) {
        $selected_room = $row;
    }
}

$booking_success = false;

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $name = $conn->real_escape_string($_POST['name']);
    $phone = $conn->real_escape_string($_POST['phone']);
    $email = $conn->real_escape_string($_POST['email']);
    $room_id = $_POST['room_id'];
    $check_in = $_POST['check_in'];
    $check_out = $_POST['check_out'];
    $total_price = $_POST['total_price'];

    // Check availability
    $stmt = $conn->prepare("SELECT COUNT(*) FROM booking WHERE room_id = ? AND status IN ('pending', 'confirmed') AND ((check_in BETWEEN ? AND ?) OR (check_out BETWEEN ? AND ?))");
    $stmt->bind_param("issss", $room_id, $check_in, $check_out, $check_in, $check_out);
    $stmt->execute();
    $stmt->bind_result($booking_count);
    $stmt->fetch();
    $stmt->close();

    if ($booking_count > 0) {
        echo "<p class='text-red-600 text-center'>Room not available for selected dates.</p>";
    } else {
        // Insert booking
        $stmt = $conn->prepare("INSERT INTO booking (user_id, name, phone, email, room_id, check_in, check_out, total_price, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending')");
        $stmt->bind_param("issssssd", $user_id, $name, $phone, $email, $room_id, $check_in, $check_out, $total_price);

        if ($stmt->execute()) {
            $booking_success = true; // ✅ Set success flag for pop-up
        } else {
            echo "<p class='text-red-600 text-center'>Error: " . $conn->error . "</p>";
        }
    }
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <title>Book a Stay | Bungoma Resort</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://kit.fontawesome.com/a076d05399.js" crossorigin="anonymous"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <style>
        body {
            background: url('assets/images/hero.jpg') center/cover no-repeat;
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        /* Success Popup Styling */
        .popup {
            display: none;
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.3);
            text-align: center;
            width: 300px;
        }
        .popup i {
            font-size: 50px;
            color: green;
        }
    </style>
</head>
<body>

    <!-- Floating Booking Card -->
    <div class="bg-white p-8 rounded-lg shadow-2xl w-[400px]">
        <h2 class="text-2xl font-bold text-center mb-6">Book Your Stay</h2>
        
        <form method="POST" id="booking-form">
        <div class="mb-4">
    <label class="block font-medium mb-1"><i class="fas fa-user"></i> Full Name:</label>
    <input type="text" name="name" class="w-full p-3 border rounded-lg focus:ring-2 focus:ring-blue-500" 
        value="<?= htmlspecialchars($user['name'] ?? '') ?>" required>
</div>

<div class="mb-4">
    <label class="block font-medium mb-1"><i class="fas fa-phone"></i> Phone:</label>
    <input type="text" name="phone" class="w-full p-3 border rounded-lg focus:ring-2 focus:ring-blue-500" 
        value="<?= htmlspecialchars($user['phone'] ?? '') ?>" required>
</div>

<div class="mb-4">
    <label class="block font-medium mb-1"><i class="fas fa-envelope"></i> Email:</label>
    <input type="email" name="email" class="w-full p-3 border rounded-lg focus:ring-2 focus:ring-blue-500" 
        value="<?= htmlspecialchars($user['email'] ?? '') ?>" required>
</div>


            <div class="mb-4">
                <label class="block font-medium mb-1"><i class="fas fa-hotel"></i> Room Type:</label>
                <select name="room_id" id="room_id" class="w-full p-3 border rounded-lg focus:ring-2 focus:ring-blue-500" required>
                    <option value="" disabled>Select a room</option>
                    <?php foreach ($rooms as $room): ?>
                        <option value="<?= $room['id'] ?>" data-price="<?= $room['price'] ?>" data-room-type="<?= $room['room_type'] ?>" <?= ($selected_room_id == $room['id']) ? 'selected' : '' ?>>
                            <?= $room['name'] ?> - Ksh <?= number_format($room['price']) ?> / night
                        </option>
                    <?php endforeach; ?>
                </select>
            <div class="mb-4">
                <label class="block font-medium mb-1"><i class="fas fa-calendar-alt"></i> Check-in Date:</label>
                <input type="date" id="check_in" name="check_in" min="<?= date('Y-m-d',strtotime(date('Y-m-d')." +0 day")) ?>" class="form-control form-control-sm form-control-border" required>
            </div>

            <div class="mb-4">
                <label class="block font-medium mb-1"><i class="fas fa-calendar-alt"></i> Check-out Date:</label>
                <input type="date" id="check_out" name="check_out" min="<?= date('Y-m-d',strtotime(date('Y-m-d')." +1 days")) ?>" class="w-full p-3 border rounded-lg focus:ring-2 focus:ring-blue-500"  required>
            </div>

            <div class="mb-4 text-lg font-semibold">
                <i class="fas fa-money-bill-wave"></i> Total Price: <span id="total_price">Ksh 0</span>
                <input type="hidden" name="total_price" id="total_price_input">
            </div>

            <button type="submit" id="submit_btn" class="w-full bg-red-500 text-white py-3 rounded-lg font-bold hover:bg-red-600 transition disabled:opacity-50">
                <i class="fas fa-bed"></i> Book Now
            </button>
        </form>
    </div>

    <!-- Success Popup -->
    <div class="popup" id="successPopup">
        <i class="fas fa-check-circle"></i>
        <h2 class="text-xl font-bold mt-3">Booking Successful!</h2>
        <p class="text-gray-600">Redirecting to dashboard...</p>
    </div>
    <script>
        $(document).ready(function () {
            <?php if ($booking_success): ?>
                $("#successPopup").fadeIn();
                setTimeout(function () {
                    window.location.href = "dashboard.php";
                }, 3000);
            <?php endif; ?>
        });
        $(document).ready(function () {
    function updateRoomType() {
        let selectedRoom = $("#room_id option:selected");
        let roomType = selectedRoom.data("room-type") || "Select a room to see the type";
        $("#room_type_display").text(roomType);
    }

    function calculateTotalPrice() {
        let price = $("#room_id option:selected").data("price") || 0;
        let checkInDate = $("#check_in").val();
        let checkOutDate = $("#check_out").val();

        if (checkInDate && checkOutDate) {
            let checkIn = new Date(checkInDate);
            let checkOut = new Date(checkOutDate);

            if (checkOut > checkIn) {
                let nights = (checkOut - checkIn) / (1000 * 60 * 60 * 24);
                let total = nights * price;

                $("#total_price").text("Ksh " + total.toLocaleString());
                $("#total_price_input").val(total);
                $("#submit_btn").prop("disabled", false);
            } else {
                $("#total_price").text("Ksh 0");
                $("#total_price_input").val(0);
                $("#submit_btn").prop("disabled", true);
            }
        }
    }

    $("#room_id").change(function () {
        updateRoomType();
        calculateTotalPrice();
    });

    $("#check_in, #check_out").change(function () {
        calculateTotalPrice();
    });

    // Initialize calculations on page load
    updateRoomType();
    calculateTotalPrice();
});

    </script>

</body>
</html>
